import os
import xml.etree.ElementTree as ET
import zipfile
import glob
import re

def remove_parentheses_content(text):
    """去除文本中的括号及其内容"""
    # 移除圆括号及其内容
    text = re.sub(r'\([^)]*\)', '', text)
    # 移除方括号及其内容
    text = re.sub(r'\[[^\]]*\]', '', text)
    # 移除花括号及其内容（如果需要）
    text = re.sub(r'\{[^}]*\}', '', text)
    # 去除多余空格
    return text.strip()

def create_gamelist_from_mame():
    # 检查mame.xml是否存在
    if not os.path.exists('mame.xml'):
        print("错误: 当前目录下未找到mame.xml文件")
        return
    
    # 解析mame.xml
    try:
        mame_tree = ET.parse('mame.xml')
        mame_root = mame_tree.getroot()
    except Exception as e:
        print(f"解析mame.xml时出错: {e}")
        return
    
    # 创建machine名称到机器信息的映射
    machine_dict = {}
    for machine in mame_root.findall('machine'):
        name = machine.get('name')
        if name:
            machine_dict[name] = machine
    
    # 创建新的gameList根元素
    game_list = ET.Element('gameList')
    
    # 添加provider部分（从模板复制）
    provider = ET.SubElement(game_list, 'provider')
    ET.SubElement(provider, 'System').text = 'NEOGEO'
    ET.SubElement(provider, 'software').text = 'Google'
    ET.SubElement(provider, 'database').text = 'Google.com'
    ET.SubElement(provider, 'web').text = 'http://www.google.com'
    
    # 遍历当前目录下的所有zip文件
    zip_files = glob.glob('*.zip')
    processed_count = 0
    
    for zip_file in zip_files:
        # 获取文件名（不包括扩展名）
        base_name = os.path.splitext(zip_file)[0]
        
        # 检查是否在mame.xml中存在对应的machine
        if base_name in machine_dict:
            machine = machine_dict[base_name]
            
            # 创建game元素
            game = ET.SubElement(game_list, 'game')
            game.set('id', '109352')  # 使用固定ID或可以生成唯一ID
            game.set('source', 'Google.com')
            
            # path: ./文件名（包括扩展名）
            ET.SubElement(game, 'path').text = f'./{zip_file}'
            
            # name: description（如果包括/号，取/号后面部分），并去除括号内容
            desc_elem = machine.find('description')
            if desc_elem is not None and desc_elem.text:
                description = desc_elem.text
                if '/' in description:
                    name = description.split('/', 1)[1].strip()
                else:
                    name = description
                # 去除括号内容
                name = remove_parentheses_content(name)
            else:
                name = base_name
            ET.SubElement(game, 'name').text = name
            
            # desc: 固定为'NEOGEO Game'
            ET.SubElement(game, 'desc').text = 'NEOGEO Game'
            
            # releasedate: year0101T000000
            year_elem = machine.find('year')
            if year_elem is not None and year_elem.text:
                releasedate = f"{year_elem.text}0101T000000"
            else:
                releasedate = "19900101T000000"  # 默认值
            ET.SubElement(game, 'releasedate').text = releasedate
            
            # developer: manufacturer
            manufacturer_elem = machine.find('manufacturer')
            if manufacturer_elem is not None and manufacturer_elem.text:
                developer = manufacturer_elem.text
            else:
                developer = ""
            ET.SubElement(game, 'developer').text = developer
            
            # 其他字段（从模板复制或使用默认值）
            ET.SubElement(game, 'publisher').text = ""
            ET.SubElement(game, 'genre').text = ""
            ET.SubElement(game, 'players').text = "1-2"
            ET.SubElement(game, 'hash').text = "475215A0"  # 可以使用实际哈希值
            
            # image: ./imgs/文件名（不包括扩展名）.jpg
            ET.SubElement(game, 'image').text = f'./imgs/{base_name}.jpg'
            ET.SubElement(game, 'thumbnail').text = ""
            ET.SubElement(game, 'genreid').text = "2048"
            
            processed_count += 1
            print(f"已处理: {zip_file} -> {name}")
    
    # 保存到gamelist.xml
    try:
        # 美化XML输出
        def indent(elem, level=0):
            i = "\n" + level*"  "
            if len(elem):
                if not elem.text or not elem.text.strip():
                    elem.text = i + "  "
                if not elem.tail or not elem.tail.strip():
                    elem.tail = i
                for child in elem:
                    indent(child, level+1)
                if not elem.tail or not elem.tail.strip():
                    elem.tail = i
            else:
                if level and (not elem.tail or not elem.tail.strip()):
                    elem.tail = i
        
        indent(game_list)
        
        tree = ET.ElementTree(game_list)
        # 添加XML声明
        with open('gamelist.xml', 'wb') as f:
            f.write(b'<?xml version="1.0" encoding="utf-8" standalone="yes"?>\n')
            tree.write(f, encoding='utf-8')
        
        print(f"\n成功创建gamelist.xml！共处理了{processed_count}个游戏。")
        
    except Exception as e:
        print(f"保存gamelist.xml时出错: {e}")

if __name__ == "__main__":
    create_gamelist_from_mame()